/*---------------------------------------------------------------------------+
| PMSPYUTL.C                                                                 |
+----------------------------------------------------------------------------+
|                                                                            |
| Utility routines                                                           |
|                                                                            |
+----------------------------------------------------------------------------+
| Change History:
| ---------------
| $C1=2.00,08/09/89,smd,creation
+---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------+
| Includes                                                                   |
+---------------------------------------------------------------------------*/

#include "pmspy.h"                      /* Resource symbolic identifiers*/

#include <ctype.h>

static MEM_DATA memStats = { 0, 0, 0};

/***************************************************************************
*
* Routine:    AllocateMemory
*
* Function:   Allocate dynamic memory
*
***************************************************************************/

PVOID AllocateMemory(register USHORT uSizeWanted )

{
  PVOID p;

  /*** Attempt to allocate the requested memory ***/

  if ( (p = malloc( uSizeWanted )) == NULL)
  {

#if defined(NON_PM_TESTING)

    printf("\n\n**** Unable to allocate %u bytes\n", uSizeWanted);
    exit(1);

#else
    MsgDisplay(HWND_DESKTOP,
               Strings[IDS_TITLE],
               Strings[IDS_MSG_ALLOCATE],
               NULL,
               MB_CUAWARNING | MB_CANCEL,

               uSizeWanted,
               memStats.nAllocs,
               memStats.sAllocs,
               memStats.nFrees);

    WinPostMsg(hwndFrame, PMSPY_QUIT_NOTICE, (ULONG) NULL, (ULONG) NULL);

#endif

  }
  else
  {
    /*** Update statistics ***/

    memStats.nAllocs++;

    memStats.sAllocs += uSizeWanted;
  }

  return( p );

}

/***************************************************************************
*
* Routine:    FreeMemory
*
* Function:   Free memory obtained from AllocateMemory
*
***************************************************************************/

VOID FreeMemory(register PVOID p )

{
   if (p == NULL)
   {
#if defined(NON_PM_TESTING)

    printf("\n\n**** Unable to free %p\n", p);
    exit(1);

#else
    MsgDisplay(HWND_DESKTOP,
               Strings[IDS_TITLE],
               Strings[IDS_MSG_FREE],
               NULL,
               MB_CUAWARNING | MB_CANCEL,

               p,
               memStats.nAllocs,
               memStats.sAllocs,
               memStats.nFrees);

    WinPostMsg(hwndFrame, PMSPY_QUIT_NOTICE, (ULONG) NULL, (ULONG) NULL);

#endif
   }
   else
   {
     memStats.nFrees++;

     free( p );
   } /* endif */
}

/***************************************************************************
*
* Routine:    QueryMemory
*
* Function:   Query memory statistics
*
***************************************************************************/

VOID QueryMemory(register PMEM_DATA p )

{
  *p = memStats;                /* return COPY to caller */
}

/***************************************************************************
*
* Routine:    MsgDisplay
*
* Function:   Display MessageBox with variable substitution
*
***************************************************************************/

USHORT MsgDisplay( HWND   hWnd,
                   PSZ    strCap,
                   PSZ    strFormat,
                   USHORT mb_id,
                   USHORT style,
                   ...)

{
  CHAR          MessageText[512];                       /* Message box output           */

  va_list       arg_ptr;                                /* Variable argument list ptr   */

 /***************************************************************************************/
 /*                                                                                     */
 /*                                MAINLINE                                             */
 /*                                                                                     */
 /***************************************************************************************/

  va_start(arg_ptr, style);                             /* Start variable list access   */
  return( vsprintf(                                     /* Format the message box output*/
                   MessageText,                         /*  Buffer to put it in         */
                   strFormat,                           /*  Format string               */
                   arg_ptr) == -1                       /*  Parameters                  */

          ? MID_ERROR

          : WinMessageBox(                              /* Display message box          */
                           HWND_DESKTOP,                /*   Desktop handle             */
                           hWnd,                        /*   Client window handle       */
                           MessageText,                 /*   Main body of message text  */
                           strCap,                      /*   Caption of Box             */
                           mb_id,                       /*   ID of message box (Help)   */
                           style       |
                           MB_MOVEABLE |
                           MB_SYSTEMMODAL)      /*   Style of message box       */
        );
}

/*---------------------------------------------------------------------------+
|  Routine: ConvertToNumber                                                  |
+----------------------------------------------------------------------------+
|
| Purpose:   Convert an Ascii text string to its binary value using
|            C language format conventions
|
|            a) decimal          dddd
|            b) octal           0dddd
|            c) hexidecimal    0xdddd
|
|            Complete Format: [<white space>]<number>
|
|
| Prototype: BOOL ConvertToNumber(PSZ   pAscii,
|                                 CTN  *pNumber,
|                                 CTN   lMin,
|                                 CTN   lMax);
|
| Entry:     pAscii.....pointer to NULL terminated string to attempt to
|                       convert to binary
|            pNumeric...pointer to variable to receive converted value
|            lMim.......minimum acceptable range value (inclusive)
|            lMax.......maximum acceptable range value (inclusive)
|
| Exit:      TRUE.......conversion was successful
|                       ==> *pNumber contains converted value
|
|            FALSE......conversion was not successful
|                       ==> *pNumber is indeterminate
|
+---------------------------------------------------------------------------*/

BOOL ConvertToNumber(PSZ     pAscii,
                     PUSHORT pNumber,
                     USHORT  lMin,
                     USHORT  lMax)

{
  register BOOL  fOK = FALSE;                 // start as a pessimist!
           PSZ   pEnd;                        // locates last char converted

           ULONG ulConversion = strtoul(pAscii,
                                        &pEnd,
                                        0);     // figure base out

  /*********************************************************************
  * Determine if conversion was successful
  *********************************************************************/

  if (*pEnd == NULL)
  {
    /*********************************************************************
    * Conversion was successful, is number within allowed range?
    *********************************************************************/

    *pNumber = LOUSHORT(ulConversion); // convert from 32 to 16 bits

    if ( (*pNumber >= lMin) && (*pNumber <= lMax) )
      fOK = TRUE;
  }

  return( fOK );
}

/*---------------------------------------------------------------------------+
|  Routine: OpenPDTR
+----------------------------------------------------------------------------+
|
| Purpose:   Open a Programmer Defined Text Resource (PDTR) for later READing
|
| Prototype: PPDTR_DATA openPDTR(PPDTR_DATA pData)
|
| Entry:     pData......pointer to PDTR control structure
|
| Exit:      !NULL......Open was successful
|                       ==> ready to Read/Eof/Close
|
|             NULL......Open failed
|                       ==> probable causes: - invalid hModule, typePDTR,
|                                              or idPDTR value specified
|
+--------------------------------------------------------------------------*/

PPDTR_DATA OpenPDTR(register PPDTR_DATA pData)

{
  register BOOL bOK = FALSE;

  if (DosGetResource(  pData->hModule,         /* Who to get from */
                       pData->typePDTR,        /* What Resource Type */
                       pData->idPDTR,          /* What Resource ID */
                     &(pData->selPDTR)) == 0)  /* Where placed */
  {
    pData->pTOF = pData->pRead = MAKEP(pData->selPDTR, 0);

    bOK = TRUE;
  }

  return(bOK ? pData : NULL);
}

#define PDTR_EOF 0x1A                  /* End-Of-File byte */
#define PDTR_EOL 0xD                   /* End-Of-Line byte */

/*---------------------------------------------------------------------------+
|  Routine: EofPDTR
+----------------------------------------------------------------------------+
|
| Purpose:   Determine if Open Programmer Defined Text Resource (PDTR)
|            is at End-Of-File
|
| Prototype: BOOL EofPDTR(PPDTR_DATA pData)
|
| Entry:     pData......pointer to PDTR control structure
|
| Exit:      TRUE.......File is at EOF
|
|            FALSE......File is not at EOF
|
+--------------------------------------------------------------------------*/

BOOL EofPDTR(register PPDTR_DATA pData)

{
  return( *pData->pRead == PDTR_EOF );
}

/*---------------------------------------------------------------------------+
|  Routine: GetsPDTR
+----------------------------------------------------------------------------+
|
| Purpose:   Get String from an Open Programmer Defined Text Resource (PDTR)
|
| Prototype: PSZ GetsPDTR(PPDTR_DATA pData, PSZ pString, USHORT lString)
|
| Entry:     pData......pointer to PDTR control structure
|            pString....pointer to area to put string
|            lString....length of pString area (must be ONE or more!)
|
| Exit:      NULL.......File is at EOF
|           !NULL.......entry pString value
|
+--------------------------------------------------------------------------*/

/* PDTR format: <text....><CR><LF>
**              <text....><CR><LF>
**              <text....><CR><LF>
**              <EOF>
*/

PSZ GetsPDTR(         PPDTR_DATA pData,
                      PSZ        pString,
             register USHORT     lString)

{
  register PSZ pWork;                           /* work in pString area */

  /* if at EOF, quit right away */

  if ( *pData->pRead == PDTR_EOF )
    return( NULL );

  /* Copy line from SEGMENT until: 1) EOL reached
  **                               2) pString area filled
  */
  for(/* Init */ pWork = pString;               /* where to put string */

      /* Term */ (*pData->pRead != PDTR_EOL) && /* quit @ EOL */
                 (lString       != 1);          /* or no room left */

      /* Iter */ pWork++,                       /* move to next string place */
                 pData->pRead++,                /* move READ pointer */
                 lString--                      /* adjust remaining size */
     )
    *pWork = *pData->pRead;

  pData->pRead += 2;   /* skip CR/LF */

  *pWork = NULL;       /* make NULL terminated.... */

  return( pString );
}

/*---------------------------------------------------------------------------+
|  Routine: ClosePDTR
+----------------------------------------------------------------------------+
|
| Purpose:   Close an Open Programmer Defined Text Resource (PDTR)
|
| Prototype: BOOL ClosePDTR(PPDTR_DATA pData)
|
| Entry:     pData......pointer to PDTR control structure
|
| Exit:      0..........PDTR close successful
|
|            EOF........PDTR close unsuccessful
|
+--------------------------------------------------------------------------*/

int ClosePDTR(register PPDTR_DATA pData)

{
  return( DosFreeSeg(pData->selPDTR) == 0 ? 0 : EOF );
}

/*---------------------------------------------------------------------------+
|  Routine: FindWordInList
+----------------------------------------------------------------------------+
|
| Purpose:   Find a NULL terminated "word" in a "word list"
|
| Prototype: USHORT FindWordInList(PSZ pWord, PWORD_LIST pWordList);
|
| Entry:     pWord......pointer to WORD to look for in pWordList
|            pWordList..pointer to WORD list
|
| Exit:      'value' for "word" (or value from End-Of-List marker saying
|            that the "word" was not found
|
+--------------------------------------------------------------------------*/

USHORT FindWordInList(PSZ pWord, PWORD_LIST pWordList)

{
  for( /* Init  */   /**** N O N E ****/;

       /* While */   (pWordList->ppWord != NULL) &&
                     (strcmpi(pWord, *pWordList->ppWord) != 0);

       /* Iter  */   pWordList++ );

  return(pWordList->vWord );

}


/*---------------------------------------------------------------------------+
|  Routine: UpdateSpyTitle
+----------------------------------------------------------------------------+
|
| Purpose:   Update the title of the PMSPY frame
|
| Prototype: VOID UpdateSpyTitle( HWND         hwndFrame,
|                                 SPY_TITLES   whichTitle,
|                                 ...);
|
| Entry:     hwndFrame....frame window
|            whichTitle...desired title mode
|
| Exit:      NONE, except title in frame updated
|
+--------------------------------------------------------------------------*/

VOID UpdateSpyTitle(HWND         hwndFrame,
                    SPY_TITLES   whichTitle,
                    ...)
{
  CHAR          szText[256];

  va_list       arg_ptr;           // Variable argument list ptr

  /*********************************************************************
  * Locate start of variable parm(s) passed to us
  *********************************************************************/

  va_start(arg_ptr, whichTitle);        // Start variable list access

  /*********************************************************************
  * Generate the new PMSPY title
  *********************************************************************/
  switch( whichTitle )
  {
    /*********************************************************************
    * Display information about the "window" the POINTER is currently over
    *********************************************************************/
    case tSpyWho:
         vsprintf(szText, Strings[IDS_FMT_PTR_QUEUE], arg_ptr);
    break;

    /*********************************************************************
    * Change title to display "window" being SPYed
    *********************************************************************/
    case tWindow:
         vsprintf(szText, Strings[IDS_FMT_SPY_WINDOW], arg_ptr);
    break;

    /*********************************************************************
    * Change title to display "object" being SPYed
    *********************************************************************/
    case tObject:
         vsprintf(szText, Strings[IDS_FMT_SPY_OBJECT], arg_ptr);
    break;

    /*********************************************************************
    * Change title to display "queue" being SPYed
    *********************************************************************/
    case tQueue:
         vsprintf(szText, Strings[IDS_FMT_SPY_QUEUE], arg_ptr);
    break;

    /*********************************************************************
    * Change title back to PMSPY standard
    *********************************************************************/

    default:
         strcpy(szText, swcntrl.szSwtitle);
  }

  /*********************************************************************
  * Update the PMSPY title
  *********************************************************************/

  WinSetWindowText(hwndFrame, szText);

  /*********************************************************************
  * LOG the new PMSPY title, if we need to
  *********************************************************************/

  if ( bCurrentlyLogging )
    fprintf(pLogFile, "%s\n", szText);
}

/*-------------------------------------------------------------------------
|  Routine: TranslateClassName
+--------------------------------------------------------------------------
|
| Purpose:  Translate WinQueryClassName to usable WC_* string
|
| Prototype: PSZ TranslateClassName(PSZ pszClassName)
|
| Entry:     pszClassName....name from WinQueryClassName
|
| Returns:   if a standard WC_* class, then pointer to usable string
|                                      else pointer to original class name
|
+--------------------------------------------------------------------------*/

PSZ TranslateClassName(PSZ pszClassName)

{

           PSZ    pszTranslate;        // final, translated class name
  register USHORT i;

/*--------------------------------------------------------------------------
| Translate table of WinClassName value to WC_* strings
+--------------------------------------------------------------------------*/

static struct
{ PSZ pszClassName,
      pszWC;   //  WinQueryClassName    Human-usble value
} Class2WC[] = { {"#1",                 "WC_FRAME"              },
                 {"#2",                 "WC_COMBOBOX"           },
                 {"#3",                 "WC_BUTTON"             },
                 {"#4",                 "WC_MENU"               },
                 {"#5",                 "WC_STATIC"             },
                 {"#6",                 "WC_ENTRYFIELD"         },
                 {"#7",                 "WC_LISTBOX"            },
                 {"#8",                 "WC_SCROLLBAR"          },
                 {"#9",                 "WC_TITLEBAR"           },
                 {"#A",                 "WC_MLE"                },

                 {"#10",                "WC_APPSTAT"            },
                 {"#11",                "WC_KBDSTAT"            },
                 {"#12",                "WC_PECIC"              },
                 {"#13",                "WC_DBE_KK_POPUP"       },

                 {"#32766",             "Desk Top"              },

                 {NULL,                 NULL                    }
               };

  /*********************************************************************
  * Translate all WC_* class names to something human understandable
  *********************************************************************/

  for(/* Initialize */ pszTranslate =  NULL,
                       i = 0;
      /* While      */ pszTranslate == NULL;
      /* Iterate    */ i++)
  {
     /*********************************************************************
     * At end of translate table?  if so, use class name ASIS
     *********************************************************************/
     if (Class2WC[i].pszClassName == NULL)
       pszTranslate = pszClassName;
     /*********************************************************************
     * Is this a WC_* class?  if so, use translation
     *********************************************************************/
     else if (strcmpi(pszClassName, Class2WC[i].pszClassName) == 0)
       pszTranslate = Class2WC[i].pszWC;
  }

  /*********************************************************************
  * Return the translation result
  *********************************************************************/

  return( pszTranslate );
}
